import java.rmi.*;
import java.util.*;
import java.rmi.registry.*;

/**
 * Classe che implementa l'interfaccia INewsgroupServer e IServer
 * @author Letizia Cheng Cheng Sun
 * @version 1.0
 */

public class NewsgroupServerImpl extends  java.rmi.server.UnicastRemoteObject 
								implements INewsgroupServer, IServer{
	
	/**
	 * Classe che serve solamente per la sincronizzazione e la possibilit
	 * di cambiare qualcosa nel ring
	 */
	class Token {
		public boolean changeRing;
		Token(){changeRing=false;}
	}

	//Dichiarazione di un'istanza della classe Token, usata per sincronizzazione
	protected Token tk;
	
	//Servizio che ritorna ID univoci di messaggi
	protected IIDUnique idUnique;
	
	//Bacheca locale di ogni servitore, contiene l'albero con tutti i messaggi
	protected Showcase myShowcase;
	
	//Vettore contenente la lista dei Server attivi e l'ordine del ring di 
	//servitori quando comunicheranno
	protected Vector activeServer;
	
	//Riferimento a una classe che implementa IShowMessage e che si occupa
	//dell'output dell'applicazione
	protected IShowMessage show;
	
	//Nome del server
	protected String name;
	
	//Informazioni dei registry a cui il server  collegato
	protected RegistryInfo reg;
	
	/**
	 * Costruttore di NewsgroupServerImpl con 5 parametri
	 * @param n nome del server
	 * @param id riferimento al servizio remoto che fornisce id univoci
	 * @param s riferimento a una classe che implementa l'interfaccia IShowMessage
	 * 			che gestisce l'io
	 * @param mt l'albero dei messaggi iniziali propri del server
	 * @param first true se  la prima istanza di server false altrimenti
	 * @throws RemoteException
	 */
	public NewsgroupServerImpl(String n,IIDUnique id,IShowMessage s,MsgTree mt,boolean first)
					throws RemoteException{
		super();
		this.tk=new Token();
		this.show=s;
		this.name=n;
		this.idUnique=id;
		this.myShowcase=new Showcase(mt);
		this.activeServer=new Vector();
		if(first)
			this.activeServer.add(this);
			
		this.reg=null;
	}
	
	
	
		
	/**
	 * Metodo per registare un nuovo cliente che verr servito dal server
	 * @param client_ref riferimento al cliente da registrare
	 */
	public synchronized void registerNews(INewsgroupClient client_ref)throws RemoteException{
		client_ref.callback(client_ref.who().getNickname()+" connected to "+
				name+".\n");
	}
	
	/**
	 * Metodo per cancellare la registrazione tra un cliente e questo server
	 * @param client_ref riferimento al cliente da cancellare
	 */
	public synchronized void unregisterNews(INewsgroupClient client_ref)throws RemoteException{
		client_ref.callback("Exit of "+client_ref.who().getNickname()+
				"from "+name+".\nGood bye.\n\n");
	}
	
	/**
	 * Metodo che permette di trovare tutti i messaggi con un dato oggetto e 
	 * comunicarli al client tramite callback
	 * @param client_ref riferimento al cliente che ha fatto la richiesta di lettura
	 * @param obj stringa rappresentante l'oggetto dei messaggi che si stanno cercando
	 */
	public void read(INewsgroupClient client_ref,String obj)throws RemoteException{
		Msg []mess=this.myShowcase.readMsgs(obj);
		if(mess!=null)
			client_ref.callback(mess);
		else
			client_ref.callback("Non  stato trovato alcun messaggio con questo criterio di ricerca");
		
	}
	
	/**
	 * Metodo che permette di trovare tutti i messaggi con un dato oggetto e
	 * e un dato mittente. Il metodo permette inoltre di comunicare i messaggi 
	 * al client tramite callback
	 * @param client_ref riferimento al cliente che ha fatto la richiesta di lettura
	 * @param sender mittente dei messaggi di interesse
	 * @param obj stringa rappresentante l'oggetto dei messaggi che si stanno cercando
	 */
	public void read(INewsgroupClient client_ref,Person sender,String obj)
					throws RemoteException{
		
		Msg []mess=this.myShowcase.readMsgs(sender,obj);
		if(mess!=null)
			client_ref.callback(mess);
		else
			client_ref.callback("Non  stato trovato alcun messaggio con questo criterio di ricerca");
		
	}
	
	/**
	 * Metodo che permette di trovare tutti i messaggi con un dato oggetto,
	 * mittente ed orario del messaggio. 
	 * La comunicazione col client avviene tramite callback
	 * @param client_ref riferimento al cliente che ha fatto la richiesta di lettura
	 * @param sender mittente dei messaggi di interesse
	 * @param obj stringa rappresentante l'oggetto dei messaggi che si stanno cercando
	 * @param time orario dei messaggi interessanti
	 */
	public void read(INewsgroupClient client_ref,Person sender,String obj,Date time)
				throws RemoteException{
		
		Msg []mess=this.myShowcase.readMsgs(sender,obj,time);
		if(mess!=null)
			client_ref.callback(mess);
		else
			client_ref.callback("Non  stato trovato alcun messaggio con questo criterio di ricerca");
		
	}
	
	/**
	 * Metodo che legge tutti i messaggi che non sono risposta di nessuno
	 * @param client_ref riferimento al cliente per fare la callback
	 */
	public void read(INewsgroupClient client_ref)throws RemoteException {
		Msg []mess=this.myShowcase.readAll();
		if(mess!=null)
			client_ref.callback(mess);
		else
			client_ref.callback("Non sono presenti messaggi");
	}
	
	/**
	 * Metodo che permette di trovare un determinato messaggio 
	 * tramite il suo ID univoco. 
	 * La comunicazione col client avviene tramite callback
	 * @param client_ref riferimento al cliente che ha fatto la richiesta di lettura
	 * @param ID stringa contenente l'id univoco del messaggio ricercato 
	 */
	public void readIDMsg(INewsgroupClient client_ref,String ID)throws RemoteException{
		Msg mess=this.myShowcase.readMsg(ID);
		if(mess!=null)
			client_ref.callback(mess);
		else
			client_ref.callback("Messaggio non trovato");
			
	}
	
	/**
	 * Metodo che permette di leggere tutte le risposte a un dato messaggio
	 * caratterizzato dal suo ID univoco.
	 * Comunicazione col client tramite callback
	 * @param client_ref riferimento al client remoto che ha fatto la richiesta di lettura
	 * @param ID ID del messaggio di cui si stanno cercando i messaggi risposta
	 * 
	 */
	public void readResponse(INewsgroupClient client_ref,String ID)throws RemoteException{
		Msg []mess=this.myShowcase.readResponse(ID);
		if(mess!=null)
			client_ref.callback(mess);
		else
			client_ref.callback("Non sono state trovate risposte per il messaggio indicato");
	}
	
	
	/**
	 * Metodo che crea un messaggio dai dati forniti dal cliente, inserisce il
	 * messaggio in bacheca e comunica agli altri server attivi il nuovo messaggio.
	 * Il messaggio viene creato lato server, perch il server ha bisogno di 
	 * un identificativo univoco per i messaggi e il client non deve preoccuparsi
	 * di fornirne lui uno univoco.
	 * @param client mittente del messaggio
	 * @param obj stringa contenente l'oggetto del messaggio
	 * @param body corpo del messaggio
	 * @param father riferimento al messaggio di cui  risposta, 
	 * 		se non  risposta pu essere a null
	 * @param client_ref riferimento al client per fare la callback 
	 */
	public void writeMsg(Person client,String obj,String body,
					Msg father,INewsgroupClient client_ref)throws RemoteException{
		Msg mess=new Msg(client,obj,body,this.idUnique.nextName("msg"));
		this.writeMsg(mess,father,client_ref);
		
	}
	
	/**
	 * Metodo che crea un messaggio dai dati forniti dal cliente(tra cui anche 
	 * la data), inserisce il messaggio in bacheca e 
	 * comunica agli altri server attivi il nuovo messaggio.
	 * Il messaggio viene creato lato server, perch il server ha bisogno di 
	 * un identificativo univoco per i messaggi e il client non deve preoccuparsi
	 * di fornirne lui uno univoco.
	 * @param client mittente del messaggio
	 * @param time data del messaggio
	 * @param obj stringa contenente l'oggetto del messaggio
	 * @param body corpo del messaggio
	 * @param father riferimento al messaggio di cui  risposta, 
	 * 		se non  risposta pu essere a null
	 * @param client_ref riferimento al client per fare la callback 
	 */
	public void writeMsg(Person client,Date time,String obj,String body,
			Msg father,INewsgroupClient client_ref)throws RemoteException{
		Msg mess=new Msg(client,time,obj,body,this.idUnique.nextName("msg"));
		this.writeMsg(mess,father,client_ref);

	}
	
	/**
	 * Metodo per poter vedere esternamente il nome di questo servitore
	 * @return nome del servitore
	 */
	public String getName()throws RemoteException{
		return name;
	}
	
	//-----------------------------------------------------------------------------
	//Implementazione dei metodi dell'interfaccia IServer legati al binding
	//-----------------------------------------------------------------------------
	
	/**
	 * Metodo per inserire un nuovo servitore nel ring
	 * Il metodo fa il binding con il registro di secondo livello
	 * Aggiunge al vettore di servitori attivi il riferimento al nuovo servitore,
	 * in ultima posizione e avvisa il servitore attivo successivo,
	 * che inizia una serie di chiamate successive fino ad avvisare tutti
	 * i componenti del ring
	 * @param server_ref riferimento al server che si vuole attivare
	 * @param reg Informazioni dei registri a cui si si leger poi server_ref
	 */
	public  void bindService(IServer server_ref,RegistryInfo reg)throws RemoteException{
		//TODO togliere stringhe di debug qui e nei metodi che questo chiama
		String registryURL = "rmi://"+reg.getHost()+":"+ 
						RegistryInfo.getSecondLevelPortNum()+"/"+ 
						server_ref.getName();
		this.reg=reg;
		try{
			Naming.rebind(registryURL, server_ref);
			while (!this.canChangeRing());
			this.activeServer.add(server_ref);
			while(!this.isNextServerAlive());
			this.getNextServer().insertNewServer(server_ref,this.activeServer);
		}
		catch(Exception re){
			return ;
		}
	}
	
	/**
	 * Metodo che slega un servitore dall'erogazione del servizio, 
	 * togliendosi dal registry e dal ring, avvisando i servitori 
	 * attivi successivi in sequenza
	 * @param server_ref  il riferimento del server che vuole fare l'unbind
	 * @return true se l'unibnding ha successo,false altrimenti
	 */
	public boolean unbindService(INewsgroupServer server_ref,RegistryInfo r)throws RemoteException{
		try{
			String registryURL = "rmi://"+r.getHost()+":"+ 
			RegistryInfo.getSecondLevelPortNum()+"/"+ 
			server_ref.getName();
			Naming.unbind(registryURL);
			while (!this.canChangeRing());
			int servIndex=this.activeServer.indexOf(server_ref);
			this.activeServer.remove(server_ref);
			this.getNextServer().removeServer(this.activeServer.size()+1,servIndex);
			return true;
		}
		catch(Exception e){
			return false;
		}
		
	}
	
	
	
	//---------------------------------------------------------------------------------
	//Metodi dell'interfaccia IServer atti per la comunicazione dei Server
	//------------------------------------------------------------------------------
	
	/**
	 * Metodo che serve per comunicare al server successivo nel ring l'entrata
	 * di un nuovo elemento server in coda al ring
	 * Ogni server si occupa di avvisare il proprio successivo fino alla fine del ring
	 * @param server_ref riferimento al server che fa la comunicazione
	 * @param as Vettore contenente i vettori attivi
	 */
	public void insertNewServer(IServer server_ref, Vector as)throws RemoteException{
		if(!(server_ref.getName().equals(this.name))){
			this.activeServer.add(server_ref);
			while(!this.isNextServerAlive());
			this.getNextServer().insertNewServer(server_ref,as);
		}
		else{
			this.setActiveServer(as);
			this.tk.changeRing=false;
			this.getNextServer().changeRingDone(this);
		}
	}
	
	/**
	 * Metodo per comunicare agli altri server attivi la rimozione di un server
	 * dal ring. Metodo a comunicazioni successive
	 * @param ringEl numero di elementi del ring prima della rimozione
	 * @param remEl numero dell'elemento del ring che deve essere rimosso
	 * @throws RemoteException
	 */
	public void removeServer(int ringEl, int remEl)throws RemoteException{
		if((this.activeServer.size()==ringEl)){
			this.activeServer.remove(remEl);
			this.getNextServer().removeServer(ringEl,remEl);
		}
		else{
			tk.changeRing=false;
			this.getNextServer().changeRingDone(this);
		}
	}
	
	/**
	 * Metodo per comunicare agli altri server del ring l'aggiunta di un nuovo 
	 * messaggio in bacheca, in modo da rendere il tutto consistente. La comunicazione
	 * viene fatta all'elemento successivo del ring e si incarica questo di 
	 * comunicarlo poi al proprio successivo. 
	 * @param mess nuovo messaggio, quello che deve essere inserito.
	 * @param father eventuale messaggio di cui mess  risposta, se
	 * 		mess non era una risposta father  a null
	 * @param firstServName nome univoco del server che ha iniziato la transmit.
	 * 			Serve per il controllo di terminazione del transmit nel ring
	 */
	public void transmitMsg(Msg mess,Msg father,String firstServName)throws RemoteException{
		if(!(this.name.equals(firstServName))){
			this.myShowcase.insertMsg(mess,father);
			while(!this.isNextServerAlive());
			this.getNextServer().transmitMsg(mess,father,firstServName);
		}
	}
	
	
	/**
	 * Metodo che serve a comunicare agli altri server attivi l'intenzione a cambiare qualcosa
	 * nel ring ed ottenere l'autorizzazione a farlo.
	 * Se il metodo ritorna true il server attuale pu cambiare il ring, false
	 * se non pu farlo
	 * @param s riferimento al server chiamante, per avere un'indicazione di quando si  finito
	 * 		di girare nel ring
	 * @return true se il server chiamante pu cambiare il ring, false altrimenti
	 */
	public boolean changeRing(IServer s)throws RemoteException{
		//condizione per cui ho finito di girare il ring. 
		//Se ci sono arrivata non ci sono stati intoppi, posso cambiare il ring  
		if(this.getName().equals(s.getName()))
			return true;
		
		//sincronizzo l'accesso alla variabile
		synchronized(tk){
			//Se anche questo server ha gi chiesto di avere il ring
			//non posso permettere ad un altro di cambiare il ring
			if(this.tk.changeRing) return false;
			this.tk.changeRing=true;
		}
		
		//Comunico agli altri server
		 boolean ok=this.getNextServer().changeRing(s);
		 
		 //Se gli altri server attivi permettono al chiamante di cambiare il ring
		 //ritorno true
		 if(ok)
		 	return true;
		 //Altrimenti il chiamante non pu cambiare il ring e quindi
		 //prima di ritornare false il server deve rimettere la propria variabile a false
		 this.tk.changeRing=false;
		 return false;
	}
	
	/**
	 * Metodo per liberare il token e permettere ad altri processi e/o server 
	 * ad attuare altri cambiamenti all'interno del ring, dal momento che 
	 * questi devono avvenire in mutua esclusione
	 * @param s server che aveva il token e che inizia a liberare il ring
	 */
	public void changeRingDone(IServer s)throws RemoteException{
		if(this.getName().equals(s.getName()))
			return;
		
		if(this.tk.changeRing){
			this.tk.changeRing=false;
		}
		this.getNextServer().changeRingDone(s);
	}
	
	
	//--------------------------------------------------------------------------
	//Metodi propri della classe
	//--------------------------------------------------------------------------
	
	/**
	 * Metodo per aggiornare i server attivi al momento dell'entrata nel gruppo
	 * @param as Vector di server attivi
	 */
	protected void setActiveServer(Vector as){
		this.activeServer=as;
		this.tk.changeRing=true;
	}
	
	/**
	 * Metodo che dato il messaggio gi creato lato server, si occupa di inserirlo
	 * nella bacheca, comunicare tramite callback al client se l'inserimento ha 
	 * avuto successo e comunicare l'inserimento del messaggio al server dopo
	 * questo server nel ring. Si utilizza qui una politica ottimista, in cui
	 * si comunica al client che l'inserimento ha avuto successo prima di iniziare
	 * a comunicare il cambiamento agli altri server
	 */
	protected void writeMsg(Msg mess,Msg father,INewsgroupClient client_ref)throws RemoteException{
		if(myShowcase!=null){
			if(this.myShowcase.insertMsg(mess,father)){
				client_ref.callback("Inserimento del messaggio ok.\n");
				if(this.activeServer.size()>1)
					this.getNextServer().transmitMsg(mess,father,this.name);
			
			}
			else 
				client_ref.callback("Inserimento del messaggio fallita.\n");
		}
		else client_ref.callback("Inserimento del messaggio fallita.\n");
	}
	
	/**
	 * Metodo che serve per sapere se altri nel ring stanno tentando di
	 * cambiare qualcosa nel ring. 
	 * @return se  possibile eseguire l'inserimento di un nuovo server nel ring
	 */
	protected boolean canChangeRing()throws RemoteException{
		//Sincronizzo perch non si cambi concorrentemente la
		//changeRing
		synchronized(tk){
			if(this.tk.changeRing) {
				try{
					//per evitare che entrambi liberino e basta, e per 
					//cercare di limitare le collisioni aspetto un tempo casuale
					long randomNum=Math.round(Math.random()*300);
					Thread.sleep(randomNum);
				}catch(Exception e){}
				return false;
			}
			this.tk.changeRing=true;
		}
		//controllo che gli altri server attivi diano il consenso ad eseguire
		//il cambio di ring
		boolean ok=this.getNextServer().changeRing(this);
		
		//Se ok  true significa che posso cambiare il ring
		if(ok)
			return true;
		
		//ok era false, quindi libero la variabile per chi vuole prendere il ring
		this.tk.changeRing=false;
		
		try{
			//per evitare che entrambi liberino e basta, e per 
			//cercare di limitare le collisioni aspetto un tempo casuale
			long randomNum=Math.round(Math.random()*300);
			Thread.sleep(randomNum);
		}
		catch(Exception e){}
		//return false perch il ring sembra occupato
		return false;
	}
	
	
	
	/**
	 * Il metodo ritorna il server che nel ring risulta essere situato 
	 * dopo quello della istanza chiamante
	 * @return il riferimento al server successivo all'istanza chiamante 
	 * 			secondo il ring 
	 */
	protected IServer getNextServer(){
		int myPos=this.activeServer.indexOf(this);
		IServer nextServer;
		if(myPos+1<this.activeServer.size())
			nextServer=(IServer)this.activeServer.get(myPos+1);
		else 
			nextServer=(IServer)this.activeServer.get(0);
		return nextServer;
	}
	
	/**
	 * Metodo che viene invocato per sapere se il server successivo nel ring
	 *  ancora attivo. Viene invocato durante le comunicazioni tra server
	 * dello stesso servizio. Se il server successivo non  attivo, viene 
	 * invocato il metodo protected cleanInactiveServer(), che si occupa 
	 * di eliminare tutti i server inattivi
	 * @see cleanInactiveServer()
	 * @return true se l'anello successivo del ring  attivo, false altrimenti
	 */
	protected boolean isNextServerAlive(){
		IServer nextServer=this.getNextServer();
		try{
			//Se il server successivo  ancora vivo torna true e ha finito
			nextServer.getName();
			return true;
		}
		catch(Exception e){
			//Il server successivo  inattivo, tolgo tutti i server inattivi
			//e ritorno false
			this.cleanInactiveServer();
			return false;
		}
	}
	
	/**
	 * Metodo che elimina tutti i server inattivi dal ring e fa l'unbind 
	 * con il registry. Per fare ci, controlla prima tutti i server 
	 * per vedere se sono vivi e poi fa l'unbinding
	 * di tutti gli altri confrontando i nomi del server con i servizi
	 * registrati nel registry di secondo livello, valori che ottengo
	 * tramite una list()
	 *
	 */
	protected void cleanInactiveServer(){
		try{
			if(this.reg!=null){
				Registry servReg=LocateRegistry.getRegistry(RegistryInfo.getSecondLevelPortNum());
				//Con questa ottengo i servizi registrati al Registry
				String []servName=servReg.list();
				//Array in cui verranno memorizzati i nomi dei server
				//effettivamente attivi
				String []aliveServer=new String[this.activeServer.size()];
				//Array che conterr i server non pi attivi e quindi da eliminare
				String []delete=new String[this.activeServer.size()];
				
				//Inizializzo gli array
				for(int i=0;i<this.activeServer.size();i++){
					aliveServer[i]=null;
					delete[i]=null;	
				}
				
				//FASE 1:
				//Trovo i server effettivamente attivi ed
				//elimino dal ring i server non attivi
				for(int i=0,j=0;i<this.activeServer.size();i++){
					try{
						IServer sv=(IServer)this.activeServer.get(i);
						aliveServer[j]=sv.getName();
						j++;
					}catch(Exception e2){
						//Se sono nel catch il server non  attivo e 
						//quindi lo elimino dal ring
						while (!this.canChangeRing());
						this.activeServer.remove(i);
						this.getNextServer().removeServer(this.activeServer.size()+1,i);
						
					}
				}
				
				//FASE 2:
				//Elimino i server non attivi dal registry
				
				//per sapere l'effettivo numero di 
				//server attivi e ridurre il ciclo successivo
				int numAliveSer=0;
				for(int i=0;i<servName.length;i++)
					if(servName[i]!=null) numAliveSer++;
					else break;
				
				//Costruzione dell'array con i nomi dei server da eliminare
				for(int i=0,k=0;i<servName.length;i++){
					int j=0;
					for(j=0;j<numAliveSer;j++){
						if(servName[i].equals(aliveServer[j]))
							break;
					}
					if((j==numAliveSer)&&(!servName[i].equals(aliveServer[j-1]))){
						delete[k]=servName[i];
						k++;
					}
				}
				
				//Eliminazione dei nomi contenuti nell'array delete
				//dal regitry
				try{
					for(int i=0;i<delete.length;i++)
						if(delete[i]!=null)
							Naming.unbind(delete[i]);
						else break;
				}
				catch(Exception e3){}
			}
		}
		catch(Exception e1){
			
		}	
		
	}
}

